#!/usr/bin/perl
#=======================================================================
# FileName : AileHttpCtrlCom.pl
# Summary  : HttpコマンドコントローラーAPI
# ----------------------------------------------------------------------
# DATE       NAME               REASON
# ---------- ------------------ ----------------------------------------
# 2015/03/06 Kiyoshi Oishi      Initial Release
#=======================================================================
use strict;
use CGI;
use JSON;
use Encode;
use Data::Dumper;
use IO::Socket::INET;
use Socket;

use lib '../com-lib';
use ComConst;
use ComFunc;
use CmdComFunc;
use ComKeyEventFunc;
use ComLog;
use ComKeyConf;

my $rc = ComConst::STS_OK;

$rc = &main();

exit $rc;

#=======================================================================
# FuncName : main
# Summary  : Httpコマンドコントローラー
# Argument : (IN)  $reqFile  : リクエストファイル名(scalar)
#          : (IN)  $resFile  : レスポンスファイル名(scalar)
# Return   : 正常:0, 異常:-1
# ----------------------------------------------------------------------
# DATE       NAME               REASON
# ---------- ------------------ ----------------------------------------
# 2015/03/06 Kiyoshi Oishi      Initial Release
#=======================================================================
sub main {
    my $rc = ComConst::STS_OK;
    my %conf = ();
    my %messages = ();
    my $cgi = new CGI;
    my %request = ();
    my %res_data  = (ComConst::JSON_KEY_STS=>ComConst::STS_OK, ComConst::JSON_KEY_MSG=>"", ComConst::JSON_KEY_DAT=>());
    my $resData   = "";
    my $header    = "";
    my $postData  = "";
    my $basicAuth = "";
    my $type      = "";
    my $sendkey   = "";
    my $value     = "";
    my $required_request_param = undef;
    my $definedValueVar = "";
    my $data_value      = "";
    my %procResponse = (
        status  => ComConst::STS_OK, 
        data    => (),
        message => ""
    );
    my %option_params = ();

    # 初期化処理
    if ($rc == ComConst::STS_OK) {
        $rc = ComFunc::initializeApi($0, \%conf, \%messages);
    }

    # 開始ログ出力
    ComLog::log_output(ComLog::LOG_DEBUG, $messages{CM000D001});
    
    # ディスプレイの設定をしないとキーが送れないので、設定する
    $ENV{DISPLAY} = ComConst::DISPLAY;

    # リクエストデータの取得
    if ($rc == ComConst::STS_OK) {
        $rc = ComFunc::getCtrlAipRequestData(\$cgi, \%request);
        if ($rc != ComConst::STS_OK) {
            $res_data{ComConst::JSON_KEY_STS} = ComConst::STS_NG;
        }
    }

    # リクエストパラメータチェック
    if ($rc == ComConst::STS_OK) {
        $type    = defined($request{type}) ? $request{type} : "";
        $sendkey = defined($request{sendkey}) ? ucfirst(lc($request{sendkey})) : "";
        $required_request_param = $ComConst::RequiredRequestParams->{$type}->{$sendkey};
        if (defined($required_request_param)) {
            # 必須パラメータが全てあるかチェック
            for my $key (@{$required_request_param->{required}}) {
                if (!defined($request{$key}) || $request{$key} eq "") {
                    $rc = ComConst::STS_NG;
                    last;
                }
            }
            if ($rc == ComConst::STS_OK) {
                # 必須パラメータが全てある場合は、使用する値を取得
                my $valueKey = $required_request_param->{value_param_key};
                my $format = $required_request_param->{format};
                if ($format ne "") {
                    $value = sprintf($format, $request{$valueKey});
                } else {
                    $value = $request{$valueKey};
                }
                $data_value = $value;
            }
            if ($rc == ComConst::STS_OK) {
                for my $key (@{$required_request_param->{option_param}}) {
                    if (defined($request{$key})) {
                        $option_params{$key} = $request{$key};
                    } else {
                        $option_params{$key} = 1;
                    }
                }
            }
        } else {
            $rc = ComConst::STS_NG;
        }
        if ($rc != ComConst::STS_OK) {
            setResponseData(\%res_data, $rc, $messages{CM004E001});
        }
    }

    # リクエストデータチェック
    if ($rc == ComConst::STS_OK) {
        my $validate_func = $required_request_param->{validate_data_func};
        # データチェック関数が定義されている場合データ内容をチェック
        if ($validate_func ne "") {
            if (defined(&$validate_func)) {
                my $func = \&$validate_func;
                my $definedNameKey = $required_request_param->{defineValueKeyName};
                if ($definedNameKey ne "") {
                    $definedValueVar = $ComKeyConf::AileWebCmdComKey->{$definedNameKey};
                    $rc = $func->($value, $definedValueVar);
                    if ($rc == ComConst::STS_OK) {
                        $data_value = $definedValueVar->{$value};
                    }
                } else {
                    $rc = $func->($value, undef);
                }
            } else {
                $rc = ComConst::STS_NG;
            }
        } else {
            my $definedNameKey = $required_request_param->{defineValueKeyName};
            if ($definedNameKey ne "") {
                $definedValueVar = $ComKeyConf::AileWebCmdComKey->{$definedNameKey};
                $data_value = $definedValueVar->{$value};
            }
        }
        if (defined($required_request_param->{validate_option_func})) {
            my $option_validate_func = $required_request_param->{validate_option_func};
            for my $key (keys(%option_params)) {
                my $validate_func = $option_validate_func->{$key};
                if (defined(&$validate_func)) {
                    my $func = \&$validate_func;
                    $rc = $func->($option_params{$key});
                    if ($rc != ComConst::STS_OK) {
                        last;
                    }
                }
            }
        }
        if ($rc != ComConst::STS_OK) {
            setResponseData(\%res_data, $rc, $messages{CM004E001});
        }
    }

    # Basic認証データ取得
    if ($rc == ComConst::STS_OK) {
        my @httpAuth = split(/ /, $ENV{HTTP_AUTHORIZATION});
        $basicAuth = $httpAuth[1];
    }

    # リクエストデータ作成
    if ($rc == ComConst::STS_OK) {
        if ($type eq ComConst::TYPEKEYSYS) {
            $rc = ComFunc::getPostSendData(\$postData, $required_request_param->{template}, $sendkey, $data_value, "");
        } elsif ($type eq ComConst::TYPEKEYINQ) {
            if ($sendkey eq ComConst::SENDKEYVIEWLAYOUT) {
                $rc = ComFunc::getPostSendDataViewLayout(\$postData, $required_request_param->{template}, $sendkey, $data_value, $basicAuth, \%option_params);
            } else {
                $rc = ComFunc::getPostSendData(\$postData, $required_request_param->{template}, $sendkey, $data_value, $basicAuth, "");
            }
        }
    }

    # 実行コマンド作成
    if ($rc == ComConst::STS_OK) {
        # nothing
    }

    # リクエストデータをファイルに出力
    if ($rc == ComConst::STS_OK) {
        # nothing
    }

    # コマンドを呼び出し
    if ($rc == ComConst::STS_OK) {
        my $requestData = decode_json("[" . $postData . "]");
        if ($type eq ComConst::TYPEKEYSYS) {
            ComKeyEventFunc::sendKeyCode($requestData->[0]);
        } elsif ($type eq ComConst::TYPEKEYINQ) {
            aileunProcCom($requestData->[0], \%conf, \%procResponse);
            $res_data{ComConst::JSON_KEY_STS} = $procResponse{status};
            $res_data{ComConst::JSON_KEY_DAT} = $procResponse{data};
            if (-1 == $res_data{ComConst::JSON_KEY_STS}) {
                $res_data{ComConst::JSON_KEY_MSG} = $messages{$procResponse{message}};
            } else {
                $res_data{ComConst::JSON_KEY_MSG} = $procResponse{message};
            }
        }
    }

    # 結果ファイルを読み込む
    $resData = ComFunc::readResponseFile($conf{JSON_FILE}->{PROXY_RESPONSE}, \%res_data);
    ComLog::log_output(ComLog::LOG_DEBUG, Dumper($resData));

    # リクエストファイルとレスポンスファイルを削除
    ComFunc::rmJsonFile($conf{JSON_FILE});

    # 結果を返す
    $header = $cgi->header(-type=>'text/plain', -charset=>'UTF-8');
    ComFunc::setCtrlApiResponsData($resData, $header);

    # 終了ログ出力
    ComLog::log_output(ComLog::LOG_DEBUG, $messages{CM000D002});

    return $rc;
}

#=======================================================================
# FuncName : isNumValidValue
# Summary  : 数値チェック
# Argument : (IN) $value    : チェックデータ(scalar)
# Return   : 正常:0, 異常:-1
# ----------------------------------------------------------------------
# DATE       NAME               REASON
# ---------- ------------------ ----------------------------------------
# 2015/11/26 Kiyoshi Oishi      Initial Release
#=======================================================================
sub isNumValidValue {
    my $rc = ComConst::STS_OK;
    my $value = $_[0];

    if ($value eq "" || $value < 0) {
        $rc = ComConst::STS_NG;
    }

    return $rc;
}

#=======================================================================
# FuncName : isNumeric
# Summary  : 数値チェック
# Argument : (IN) $value    : チェックデータ(scalar)
# Return   : 正常:0, 異常:-1
# ----------------------------------------------------------------------
# DATE       NAME               REASON
# ---------- ------------------ ----------------------------------------
# 2015/11/26 Kiyoshi Oishi      Initial Release
#=======================================================================
sub isNumeric {
    my $rc = ComConst::STS_OK;
    my $value = $_[0];

    if ($value !~ /^\d+$/ || $value == 0) {
        $rc = ComConst::STS_NG;
    }

    return $rc;
}

#=======================================================================
# FuncName : isCamNumValidValue
# Summary  : 数値チェック
# Argument : (IN) $value    : チェックデータ(scalar)
# Return   : 正常:0, 異常:-1
# ----------------------------------------------------------------------
# DATE       NAME               REASON
# ---------- ------------------ ----------------------------------------
# 2015/11/26 Kiyoshi Oishi      Initial Release
#=======================================================================
sub isCamNumValidValue {
    my $rc = ComConst::STS_OK;
    my $value = $_[0];
    my $exec_cmd = sprintf(ComConst::GET_AILEUN_PRDCTNO_CMD, ComConst::AILEUN_PRDCTNO_FILE);
    my $prdctNo = "";


    $rc = isNumValidValue($value);
    if ($rc == ComConst::STS_OK) {
        chomp($prdctNo = `$exec_cmd`);
        my $cameraNum = $ComConst::HwType2CamNum->{$ComConst::Product2HwType->{$prdctNo}};
        if ($value > $cameraNum) {
            $rc = ComConst::STS_NG;
        }
    }

    return $rc;
}

#=======================================================================
# FuncName : isDefinedValue
# Summary  : 定義済み値チェック
# Argument : (IN) $value    : チェックデータ(scalar)
# Return   : 正常:0, 異常:-1
# ----------------------------------------------------------------------
# DATE       NAME               REASON
# ---------- ------------------ ----------------------------------------
# 2015/12/02 Kiyoshi Oishi      Initial Release
#=======================================================================
sub isDefinedValue {
    my $rc = 0;
    my $value = $_[0];
    my $permitValue = $_[1];

    if ($permitValue->{$value} eq "") {
        $rc = -1;
    }

    return $rc
}

#=======================================================================
# FuncName : setProxyRequestSetting
# Summary  : 数値チェック
# Argument : (IN) $value    : チェックデータ(scalar)
# Return   : 正常:0, 異常:-1
# ----------------------------------------------------------------------
# DATE       NAME               REASON
# ---------- ------------------ ----------------------------------------
# 2015/11/26 Kiyoshi Oishi      Initial Release
#=======================================================================
sub setProxyRequestSetting {
    my $rc = ComConst::STS_OK;
    my $baseName = $_[0];
    my $ref_conf = $_[1];

    $$ref_conf{PROXY_EXEC_CMD} = sprintf(ComConst::BASE_CMD_PATH . "/cmd_%s.sh", $baseName);
    $$ref_conf{JSON_FILE}->{PROXY_REQUEST} = sprintf(ComConst::REQUEST_PATH . "/req_%s_%s.json", $baseName, ComFunc::getPid());
    $$ref_conf{JSON_FILE}->{PROXY_RESPONSE} = sprintf(ComConst::RESPONSE_PATH . "/res_%s_%s.json", $baseName, ComFunc::getPid());

    return $rc;
}

#=======================================================================
# FuncName : setResponseData
# Summary  : レスポンスデータ設定
# Argument : (OUT) $ref_resdata  : リクエストファイル名(reference)
#          : (IN)  $res_code     : レスポンスコード(scalar)
#          : (IN)  $res_message  : メッセージ(scalar)
# Return   : Nothing
# ----------------------------------------------------------------------
# DATE       NAME               REASON
# ---------- ------------------ ----------------------------------------
# 2015/11/26 Kiyoshi Oishi      Initial Release
#=======================================================================
sub setResponseData {
    my $ref_resdata = $_[0];
    my $res_code = $_[1];
    my $res_message = $_[2];

    $ref_resdata->{ComConst::JSON_KEY_STS} = $res_code;
    $ref_resdata->{ComConst::JSON_KEY_MSG} = $res_message;
}

#=======================================================================
# FuncName : aileunProcCom
# Summary  : Aileunとの通信処理
# Argument : (IN)  $id                : リクエストID(scalar)
#          : (IN)  $ref_conf          : 設定情報(ref_associate)
#          : (OUT) $ref_proc_Response : 通信結果(ref_associate)
# Return   : Nothing
# ----------------------------------------------------------------------
# DATE       NAME               REASON
# ---------- ------------------ ----------------------------------------
# 2015/03/06 Kiyoshi Oishi      Initial Release
#=======================================================================
sub aileunProcCom {
    my $data         = $_[0];
    my $ref_conf     = $_[1];
    my $ref_procResponse = $_[2];
    my $sendMessage = "";

    # 送信メッセージ作成
    $sendMessage = sprintf(ComConst::AILEUN_SEND_MSG, $data->{btnId}, $data->{reqType}, $data->{targetView}, $data->{reqData}, $data->{auth});

    # Aileunとプロセス間通信
    tcpSocketComMain($sendMessage, $ref_conf->{TCP_PORT}, $ref_conf->{TIME_OUT}, $ref_procResponse);
}

#=======================================================================
# FuncName : tcpSocketComMain
# Summary  : TCPソケット通信処理
# Argument : (IN)  $sendMessage      : 送信メッセージ(scalar)
#          : (IN)  $tcpPort          : Tcpポート番号(scalar)
#          : (IN)  $timeOut          : タイムアウト時間(scalar)
#          : (OUT) $ref_procResponse : 通信結果(ref_associate)
# Return   : Nothing
# ----------------------------------------------------------------------
# DATE       NAME               REASON
# ---------- ------------------ ----------------------------------------
# 2015/03/06 Kiyoshi Oishi      Initial Release
#=======================================================================
sub tcpSocketComMain {
    my $sendMessage      = $_[0];
    my $tcpPort          = $_[1];
    my $timeOut          = $_[2];
    my $ref_procResponse = $_[3];
    my $server;
    my $recvMessage = "";
    my $rv;

    # ソケットの作成 & 接続
    $server = IO::Socket::INET->new(
        PeerAddr => "localhost",
        PeerPort => $tcpPort,
        Prot => "tcp",
        Timeout => $timeOut,
    );

    if (!$server) {
        $ref_procResponse->{status} = ComConst::STS_NG;
    }

    # Aileunにメッセージ送信
    if ($server) {
        if ($sendMessage eq "") {
            # 空文字だとサーバからの応答が返ってこないため
            # メッセージがからの場合は半角スペースを送信
            $sendMessage = " ";
        }
        $server->send($sendMessage);
    }

    # Aileunからの応答受信
    if ($server) {
        $server->recv($recvMessage, 4096);
        parseProcResponse($recvMessage, $ref_procResponse);
    }

    # 切断
    if ($server) {
        $server->close();
    }
}

#=======================================================================
# FuncName : parseProcResponse
# Summary  : Aileunから受信したメッセージを解析
# Argument : (IN)  $recvMessage      : 受信メッセージ(scalar)
#          : (OUT) $ref_procResponse : 通信結果(ref_associate)
# Return   : 変換後のデータ
# ----------------------------------------------------------------------
# DATE       NAME               REASON
# ---------- ------------------ ----------------------------------------
# 2015/03/12 Kiyoshi Oishi      Initial Release
#=======================================================================
sub parseProcResponse {
    my $recvMessage      = $_[0];
    my $ref_procResponse = $_[1];

    my $parseRes = decode_json($recvMessage);

    $ref_procResponse->{status}  = $parseRes->{status};
    $ref_procResponse->{data}    = $parseRes->{data};
    $ref_procResponse->{message} = $parseRes->{message};
}
