#!/usr/bin/perl
#=======================================================================
# FileName : getAileProductInfo.pl
# Summary  : Aileun製品情報取得API
# ----------------------------------------------------------------------
# DATE       NAME               REASON
# ---------- ------------------ ----------------------------------------
# 2015/02/24 Kiyoshi Oishi      Initial Release
#=======================================================================
use strict;
use CGI;
use JSON;
use Encode;
use Data::Dumper;

use lib '../com-lib';
use ComConst;
use ComFunc;
use CmdComFunc;
use ComLog;

my $rc = ComConst::STS_OK;

$rc = &main();

exit $rc;

#=======================================================================
# FuncName : main
# Summary  : Aileun製品情報取得メイン処理
# Argument : (IN)  $reqFile  : リクエストファイル名(scalar)
#          : (IN)  $resFile  : レスポンスファイル名(scalar)
# Return   : 正常:0, 異常:-1
# ----------------------------------------------------------------------
# DATE       NAME               REASON
# ---------- ------------------ ----------------------------------------
# 2015/02/24 Kiyoshi Oishi      Initial Release
#=======================================================================
sub main {
    my $rc = ComConst::STS_OK;
    my %conf = ();
    my %messages = ();
    my $cgi = new CGI;
    my %request = ();
    my %res_data = (ComConst::JSON_KEY_STS=>ComConst::STS_OK, ComConst::JSON_KEY_MSG=>"");
    my $resData = "";
    my $header = "";

    # 初期化処理
    if ($rc == ComConst::STS_OK) {
        $rc = ComFunc::initializeApi($0, \%conf, \%messages);
    }

    # 開始ログ出力
    ComLog::log_output(ComLog::LOG_DEBUG, $messages{CM000D001});

    # リクエストデータの取得
    if ($rc == ComConst::STS_OK) {
        $rc = ComFunc::getRequestData(\$cgi, \%request);
        if ($rc != ComConst::STS_OK) {
            $res_data{ComConst::JSON_KEY_STS} = ComConst::STS_NG;
        }
    }

    # リクエストデータをファイルに出力
    if ($rc == ComConst::STS_OK) {
        # nothing
    }
    
    # コマンドを呼び出し
    if ($rc == ComConst::STS_OK) {
        # Aileunのバージョン番号を取得
        $res_data{ComConst::JSON_KEY_DAT}{ProductInfo}{AileFirmVersion} = getAileFirmVersion();
        # Aileunの製品番号情報を取得
        my $productNo = getAileProductNo();
        $res_data{ComConst::JSON_KEY_DAT}{ProductInfo}{AileProductNo} = $productNo;
        # Aileunのカメラ台数情報を取得
        $res_data{ComConst::JSON_KEY_DAT}{ProductInfo}{AileCameraNum} = getAileCameraNumber($productNo);
        # Aileunの特別設定
        $res_data{ComConst::JSON_KEY_DAT}{ProductInfo}{AileSpecialConfig} = getAileSpecialConfigKey();
    }

    # 結果ファイルを読込む
    $resData = ComFunc::readResponseFile($conf{JSON_FILE}->{RESPONSE}, \%res_data);
    ComLog::log_output(ComLog::LOG_DEBUG, Dumper($resData));

    # リクエストファイルとレスポンスファイルを削除
    ComFunc::rmJsonFile($conf{JSON_FILE});

    # 結果を返す
    $header = $cgi->header(-type=>'application/json', -charset=>'UTF-8');
    ComFunc::setResponsData($resData, $header);

    # 終了ログ出力
    ComLog::log_output(ComLog::LOG_DEBUG, $messages{CM000D002});

    return $rc;
}

#=======================================================================
# FuncName : getAileFirmVersion
# Summary  : Aileunバージョン番号取得
# Argument : Nothing
# Return   : バージョン番号
# ----------------------------------------------------------------------
# DATE       NAME               REASON
# ---------- ------------------ ----------------------------------------
# 2015/02/27 Kiyoshi Oishi      Initial Release
#=======================================================================
sub getAileFirmVersion {
    my $version = "";
    my $exec_cmd = sprintf(ComConst::GET_AILEUN_VERSION_CMD, ComConst::AILEUN_VERSION_FILE);

    chomp($version = `$exec_cmd`);

    return $version;
}

#=======================================================================
# FuncName : getAileProductNo
# Summary  : Aileun製品番号取得
# Argument : Nothing
# Return   : 製品番号
# ----------------------------------------------------------------------
# DATE       NAME               REASON
# ---------- ------------------ ----------------------------------------
# 2015/02/27 Kiyoshi Oishi      Initial Release
#=======================================================================
sub getAileProductNo {
    my $prdctNo = "";
    my $exec_cmd = sprintf(ComConst::GET_AILEUN_PRDCTNO_CMD, ComConst::AILEUN_PRDCTNO_FILE);

    chomp($prdctNo = `$exec_cmd`);

    return $prdctNo;
}

#=======================================================================
# FuncName : getAileCameraNumber
# Summary  : Aileunカメラ台数取得
# Argument : Nothing
# Return   : カメラ台数
# ----------------------------------------------------------------------
# DATE       NAME               REASON
# ---------- ------------------ ----------------------------------------
# 2015/02/27 Kiyoshi Oishi      Initial Release
#=======================================================================
sub getAileCameraNumber {
    my $productNo = $_[0];
    my $cameraNum = $ComConst::HwType2CamNum->{$ComConst::Product2HwType->{$productNo}};

    return $cameraNum;
}

#=======================================================================
# FuncName : getAileSpecialConfig
# Summary  : Aileun特別設定取得
# Argument : Nothing
# Return   : 特別設定情報
# ----------------------------------------------------------------------
# DATE       NAME               REASON
# ---------- ------------------ ----------------------------------------
# 2015/02/27 Kiyoshi Oishi      Initial Release
#=======================================================================
sub getAileSpecialConfigKey {
    return ComConst::SPECIAL_CONFIG_N;
}
