#!/usr/bin/perl
package CmdComFunc;
#=======================================================================
# FileName : CmdComFunc.pm
# Summary  : コマンド側の共通メソッド
# ----------------------------------------------------------------------
# DATE       NAME               REASON
# ---------- ------------------ ----------------------------------------
# 2014/03/18 Kiyoshi Oishi      Initial Release
#=======================================================================
use strict;
use JSON;
use Encode;
use File::Basename;

use ComConst;
use ComFunc;
use CmdValidator;

#=======================================================================
# FuncName : initialize
# Summary  : 初期化処理
# Argument : (IN)  cmdName  : コマンド名 (scalar)
#          : (OUT) conf     : 設定格納用 (ref_hash)
#          : (OUT) msg      : メッセージ格納用 (ref_hash)
# Return   : 正常:0、異常:-1
# ----------------------------------------------------------------------
# DATE       NAME               REASON
# ---------- ------------------ ----------------------------------------
# 2014/03/20 Kiyoshi Oishi      Initial Release
#=======================================================================
sub initializeBaseCmd {
    my $rc = ComConst::STS_OK;
    my $cmdName  = $_[0];
    my $ref_conf = $_[1];
    my $ref_msg  = $_[2];
    my $baseName = basename($cmdName, (".pl"));
    my $cmdConfFile = sprintf("../conf/%s.conf", $baseName);
    my $msgFile = sprintf("../conf/Locale/%s/messages.properties", ComFunc::getLanguage());

    # 共通設定ファイル読込み
    ComFunc::readConfigureFile("../conf/common.conf", $ref_conf);

    # 個別設定ファイル読込み(あれば）
    if (-e $cmdConfFile) {
        ComFunc::readConfigureFile($cmdConfFile, $ref_conf);
    }

    # メッセージファイル読込み
    ComFunc::readConfigureFile($msgFile, $ref_msg);
    # ログの初期化
    ComLog::log_init($ref_conf->{LOG_LEVEL}, basename($cmdName), $ref_conf->{LOG_FILE});

    return $rc;
}

#=======================================================================
# FuncName : existFile
# Summary  : ファイル存在チェック
# Argument : Nothing
# Return   : 正常:0、異常:-1
# ----------------------------------------------------------------------
# DATE       NAME               REASON
# ---------- ------------------ ----------------------------------------
# 2014/04/10 Kiyoshi Oishi      Initial Release
#=======================================================================
sub existFile {
    my $rc = ComConst::STS_NG;
    my $targetFile = $_[0];

    # ファイル存在チェック
    if (-e $targetFile) {
        $rc = ComConst::STS_OK;
    }
    return $rc;
}

#=======================================================================
# FuncName : existWebFuncLimitFile
# Summary  : 機能制限ファイル存在チェック
# Argument : Nothing
# Return   : 正常:0、異常:-1
# ----------------------------------------------------------------------
# DATE       NAME               REASON
# ---------- ------------------ ----------------------------------------
# 2014/04/10 Kiyoshi Oishi      Initial Release
#=======================================================================
sub existWebFuncLimitFile {
    my $rc = ComConst::STS_OK;

    # ファイル存在チェック
    if (-e ComConst::FUNC_LIMITED_FILE) {
        $rc = ComConst::STS_NG;
    }
    return $rc;
}

#=======================================================================
# FuncName : checkReqData
# Summary  : 入力チェック処理
# Argument : (IN)  reqData    : リクエストデータ(scalar)
#          : (OUT) resData    : レスポンスデータ(ref_hash)
# Return   : 正常:0、異常:-1
# ----------------------------------------------------------------------
# DATE       NAME               REASON
# ---------- ------------------ ----------------------------------------
# 2014/04/10 Kiyoshi Oishi      Initial Release
#=======================================================================
sub checkReqData {
    my $rc = ComConst::STS_OK;
    my $reqData = $_[0];
    my $resData = $_[1];

    # 共通入力チェック
    my $ret = CmdValidator::comValidator($reqData->[0]);
    if (!$ret->{'state'}) {
        $rc = ComConst::STS_NG;
        $resData->{status} = ComConst::STS_NG;
        $resData->{message} = join("<br>", @{$ret->{'msg'}});
    }
    return $rc;
}

#=======================================================================
# FuncName : readRequestFile
# Summary  : リクエストファイル読み込み
# Argument : (IN) file : リクエストファイル名
# Return   : 正常:0、異常:-1
# ----------------------------------------------------------------------
# DATE       NAME               REASON
# ---------- ------------------ ----------------------------------------
# 2014/03/20 Kiyoshi Oishi      Initial Release
#=======================================================================
sub readRequestFile {
    my $file = $_[0];
    my $json = "";
    my $data = "";

    open(FH, "$file");
    $json = do {
        local $/;
        <FH>
    };
    close(FH);

    $data = decode_json($json);

    return $data;
}

#=======================================================================
# FuncName : writeResponseFile
# Summary  : レスポンスファイル出力
# Argument : (IN) file       : レスポンスファイル名
#          : (IN) ref_result : 出力データ
# Return   : 正常:0、異常:-1
# ----------------------------------------------------------------------
# DATE       NAME               REASON
# ---------- ------------------ ----------------------------------------
# 2014/03/20 Kiyoshi Oishi      Initial Release
#=======================================================================
sub writeResponseFile {
    my $rc = ComConst::STS_OK;
    my $file       = $_[0];
    my $ref_result = $_[1];
    my $json       = "";

    if ($rc == ComConst::STS_OK) {
        # ファイル名はセッションIDをつけて作成し、リクエストで一意になるようにする。
        $json = decode_utf8(encode_json($ref_result));
        open(FH, "> $file");
        print(FH $json);
        close(FH);
    }

    return $rc;
}

#=======================================================================
# FuncName : trim
# Summary  : 最初および最後から空白文字を取り除く
# Argument : (IN)  str    : 処理文字列 (scalar)
# Return   : 取り除かれた文字列 (scalar)
# ----------------------------------------------------------------------
# DATE       NAME               REASON
# ---------- ------------------ ----------------------------------------
# 2014/03/07 Kiyoshi Oishi      Initial Release
#=======================================================================
sub trim {
    my $str = $_[0];

    return ltrim(rtrim($str));
}

#=======================================================================
# FuncName : ltrim
# Summary  : 先頭からの空白文字を取り除く
# Argument : (IN)  str    : 処理文字列 (scalar)
# Return   : 取り除かれた文字列 (scalar)
# ----------------------------------------------------------------------
# DATE       NAME               REASON
# ---------- ------------------ ----------------------------------------
# 2014/03/07 Kiyoshi Oishi      Initial Release
#=======================================================================
sub ltrim {
    my $str = $_[0];

    $str =~ s/^\s+(.*)/$1/;
    return $str
}

#=======================================================================
# FuncName : rtrim
# Summary  : 最後からの空白文字を取り除く
# Argument : (IN)  str    : 処理文字列 (scalar)
# Return   : 取り除かれた文字列 (scalar)
# ----------------------------------------------------------------------
# DATE       NAME               REASON
# ---------- ------------------ ----------------------------------------
# 2014/03/07 Kiyoshi Oishi      Initial Release
#=======================================================================
sub rtrim {
    my $str = $_[0];

    $str =~ s/\s+$//;
    return $str;
}

1;
