#!/usr/bin/perl
package ComFunc;
#=======================================================================
# FileName : ComFunc.pm
# Summary  : Api側の共通メソッド
# ----------------------------------------------------------------------
# DATE       NAME               REASON
# ---------- ------------------ ----------------------------------------
# 2014/03/18 Kiyoshi Oishi      Initial Release
#=======================================================================
use strict;
use JSON;
use Encode;
use File::Basename;
use Data::Dumper;

use ComConst;
use ComLog;

#=======================================================================
# FuncName : getPid
# Summary  : プロセスID取得
# Argument : Nothing
# Return   : 正常:0、異常:-1
# ----------------------------------------------------------------------
# DATE       NAME               REASON
# ---------- ------------------ ----------------------------------------
# 2014/03/20 Kiyoshi Oishi      Initial Release
#=======================================================================
sub getPid {
    return $$;
}

#=======================================================================
# FuncName : getAileunModelNumber
# Summary  : Aileunモデル番号取得
# Argument : Nothing
# Return   : 正常:0、異常:-1
# ----------------------------------------------------------------------
# DATE       NAME               REASON
# ---------- ------------------ ----------------------------------------
# 2014/03/20 Kiyoshi Oishi      Initial Release
#=======================================================================
sub getAileunModelNumber {
    my $modelNumber = "";
    my $exec_cmd = sprintf(ComConst::GET_AILEUN_MODELNO_CMD, ComConst::AILEUN_VERSION_FILE);

    chomp($modelNumber = `$exec_cmd`);

    if ($modelNumber =~ /^[A-Z]\d$/) {
        $modelNumber =~ s/[A-Z](\d)/$1/;
    }
    $modelNumber = $modelNumber + 0;

    return $modelNumber;
}

#=======================================================================
# FuncName : getAileunUseCustomLayout
# Summary  : カスタムレイアウト設定取得
# Argument : Nothing
# Return   : 未使用:0、使用:1
# ----------------------------------------------------------------------
# DATE       NAME               REASON
# ---------- ------------------ ----------------------------------------
# 2014/03/20 Kiyoshi Oishi      Initial Release
#=======================================================================
sub getAileunUseCustomLayout {
    my $customLayoutUse = "";
    my $exec_cmd = sprintf(ComConst::GET_AILEUN_CUSTOM_LAYOUT_USE_CMD, ComConst::AILEUN_VIEW_SETTING_CONFIG);

    chomp($customLayoutUse = `$exec_cmd`);

    return $customLayoutUse;
}

#=======================================================================
# FuncName : getAileunDisplayRotate
# Summary  : 画面回転設定取得
# Argument : Nothing
# Return   : 横表示:normal、縦表示:vertical
# ----------------------------------------------------------------------
# DATE       NAME               REASON
# ---------- ------------------ ----------------------------------------
# 2014/03/20 Kiyoshi Oishi      Initial Release
#=======================================================================
sub getAileunDisplayRotate {
    my $displayRotate = "";
    my $exec_cmd = sprintf(ComConst::GET_AILEUN_DISPLAY_ROTATE_CMD, ComConst::AILEUN_DISPLAY_ROTATE_CONFIG);

    chomp($displayRotate = `$exec_cmd`);

    return $displayRotate;
}

#=======================================================================
# FuncName : getLanguage
# Summary  : 設定言語取得
# Argument : Nothing
# Return   : 正常:0、異常:-1
# ----------------------------------------------------------------------
# DATE       NAME               REASON
# ---------- ------------------ ----------------------------------------
# 2014/03/20 Kiyoshi Oishi      Initial Release
#=======================================================================
sub getLanguage {
    my $language = "";

    $language = substr($ENV{LANG}, 0, 2);
    if ($language eq "") {
        # 言語設定が取れない場合は、デフォルト「ja(英語)」
        # を言語設定にする
        $language = "ja";
    }

    return $language;
}

#=======================================================================
# FuncName : initializeApi
# Summary  : API初期化
# Argument : (IN)  apiName  : 実行API名(scalar)
#          : (OUT) ref_conf : 設定格納変数(ref_hash)
#          : (OUT) ref_msg  : メッセージ格納変数(ref_hash)
# Return   : 正常:0、異常:-1
# ----------------------------------------------------------------------
# DATE       NAME               REASON
# ---------- ------------------ ----------------------------------------
# 2014/03/20 Kiyoshi Oishi      Initial Release
#=======================================================================
sub initializeApi {
    my $rc = ComConst::STS_OK;
    my $apiName  = $_[0];
    my $ref_conf = $_[1];
    my $ref_msg  = $_[2];
    my $pid = &getPid();
    my $msgFile = sprintf(ComConst::MSG_PATH . "/%s/messages.properties", &getLanguage());
    my $baseName = ucfirst(basename($apiName, (".pl")));

    # リクエストのJSONファイル(リクエストで一意にするためPIDをファイル名に付加)
    $$ref_conf{JSON_FILE}->{REQUEST}
        = sprintf(ComConst::REQUEST_PATH . "/req_%s_%s.json", $baseName, $pid);
    # レスポンスのJSONファイル(リクエストで一意にするためPIDをファイル名に付加)
    $$ref_conf{JSON_FILE}->{RESPONSE}
        = sprintf(ComConst::RESPONSE_PATH . "/res_%s_%s.json", $baseName, $pid);
    # 実行コマンド名
    $ref_conf->{EXEC_CMD} = sprintf(ComConst::BASE_CMD_PATH . "/cmd_%s.sh", $baseName);
    

    # メッセージファイル読込み
    if ($rc == ComConst::STS_OK) {
        $rc = ComFunc::readConfigureFile($msgFile, $ref_msg);
    }

    # 設定ファイル読込み
    if ($rc == ComConst::STS_OK) {
        $rc = ComFunc::readConfigureFile(ComConst::CONF_PATH . "/common.conf", $ref_conf);
    }

    # ログの初期化
    if ($rc == ComConst::STS_OK) {
        ComLog::log_init($ref_conf->{LOG_LEVEL}, basename($apiName), $ref_conf->{LOG_FILE});
    }

    return $rc;
}

#=======================================================================
# FuncName : readConfigureFile
# Summary  : ファイル読み込み
# Argument : Nothing
# Return   : 正常:0、異常:-1
# ----------------------------------------------------------------------
# DATE       NAME               REASON
# ---------- ------------------ ----------------------------------------
# 2014/03/20 Kiyoshi Oishi      Initial Release
#=======================================================================
sub readConfigureFile {
    my $rc = ComConst::STS_OK;
    my $confFile = $_[0];
    my $ref_conf = $_[1];

    # ファイル存在チェック
    if ($rc == ComConst::STS_OK) {
        if (!-e $confFile) {
            $rc = ComConst::STS_NG;
        }
    }

    if ($rc == ComConst::STS_OK) {
        open(FH, "< $confFile");
        while (my $line = <FH>) {
            # 改行削除
            $line =~ s/\r\n$//g;
            # スペース削除
            $line = &trim($line);
            # 空行、・コメント行はスキップ
            if ($line =~ /^#|^$/) {
                next;
            }
            # 設定を「=」で分割
            my @tmp = split(/=/, $line);
            my ($key, $value) = ("", "");
            if (@tmp == 2) {
                # 値(value)に「=」が含まれない場合は分割数が2になるので
                # そのままkeyとvalueに設定
                $key = &trim($tmp[0]);
                $value = &trim($tmp[1]);
            } else {
                # 値(value)に「=」が含まれる場合は分割数が2以上になるので
                # 1つ目をkey2つ目以降をvalueとして扱うため、再度「=」で結合
                $key = &trim($tmp[0]);
                $value  = &trim(join("=", @tmp[1...$#tmp]));
            }
            # 保持用変数に格納
            $ref_conf->{$key} = $value;
        }
        close(FH);
    }

    return $rc;
}

#=======================================================================
# FuncName : readJsonFile
# Summary  : JSONファイル読み込み
# Argument : (IN) file : JSONファイル名
# Return   : 正常:0、異常:-1
# ----------------------------------------------------------------------
# DATE       NAME               REASON
# ---------- ------------------ ----------------------------------------
# 2015/03/23 Kiyoshi Oishi      Initial Release
#=======================================================================
sub readJsonFile {
    my $file = $_[0];
    my $json = "";
    my $data = "";

    open(FH, "$file");
    $json = do {
        local $/;
        <FH>
    };
    close(FH);

    $data = decode_json($json);

    return $data;
}

#=======================================================================
# FuncName : rmJsonFile
# Summary  : Jsonファイル削除
# Argument : (IN)  files : ファイル名 (ref_hash)
# Return   : 正常:0、異常:-1
# ----------------------------------------------------------------------
# DATE       NAME               REASON
# ---------- ------------------ ----------------------------------------
# 2014/03/20 Kiyoshi Oishi      Initial Release
#=======================================================================
sub rmJsonFile {
    my $rc = ComConst::STS_OK;
    my $files = $_[0];

    foreach my $key (%$files) {
        if (-e $$files{$key}) {
            unlink($$files{$key});
        }
    }

    return $rc;
}

#=======================================================================
# FuncName : getRequestData
# Summary  : リクエストデータ取得
# Argument : (IN)  ref_cgi     : CGIオブジェクト(ref)
#          : (OUT) ref_request : リクエストデータ格納用(ref_hash)
# Return   : 正常:0、異常:-1
# ----------------------------------------------------------------------
# DATE       NAME               REASON
# ---------- ------------------ ----------------------------------------
# 2014/03/20 Kiyoshi Oishi      Initial Release
#=======================================================================
sub getRequestData {
    my $rc = ComConst::STS_OK;
    my $ref_cgi     = $_[0];
    my $ref_request = $_[1];

    if ($rc == ComConst::STS_OK) {
        $$ref_request{"data"} = $$ref_cgi->param("data");
    }
    if ($$ref_request{"data"} eq "") {
        $$ref_request{"data"} = $$ref_cgi->param("POSTDATA");
    }

    return $rc;
}

#=======================================================================
# FuncName : getCtrlAipRequestData
# Summary  : リクエストデータ取得
# Argument : (IN)  ref_cgi     : CGIオブジェクト(ref)
#          : (OUT) ref_request : リクエストデータ格納用(ref_hash)
# Return   : 正常:0、異常:-1
# ----------------------------------------------------------------------
# DATE       NAME               REASON
# ---------- ------------------ ----------------------------------------
# 2014/11/27 Kiyoshi Oishi      Initial Release
#=======================================================================
sub getCtrlAipRequestData {
    my $rc = ComConst::STS_OK;
    my $ref_cgi     = $_[0];
    my $ref_request = $_[1];

    if ($rc == ComConst::STS_OK) {
        my %vars = $$ref_cgi->Vars();
        foreach my $key (keys(%vars)) {
            $$ref_request{$key} = $$ref_cgi->param($key);
        }
    }

    return $rc;
}

#=======================================================================
# FuncName : writeRequestFile
# Summary  : リクエストファイル作成
# Argument : (IN)  file        : 作成ファイル名(scalar)
#          : (IN)  ref_request : リクエストデータ(ref_hash)
# Return   : 正常:0、異常:-1
# ----------------------------------------------------------------------
# DATE       NAME               REASON
# ---------- ------------------ ----------------------------------------
# 2014/03/20 Kiyoshi Oishi      Initial Release
#=======================================================================
sub writeRequestFile {
    my $rc = ComConst::STS_OK;
    my $file        = $_[0];
    my $ref_request = $_[1];
    my $fileName    = "";

    if ($rc == ComConst::STS_OK) {
        open(FH, "> $file");
        print FH "[" . $$ref_request{"data"} . "]";
        close(FH);
    }

    return $rc;
}

#=======================================================================
# FuncName : readResponseFile
# Summary  : レスポンスファイル読み込み
# Argument : (IN)  file        : 作成ファイル名(scalar)
#          : (IN)  ref_request : リクエストデータ(ref_hash)
# Return   : 正常:0、異常:-1
# ----------------------------------------------------------------------
# DATE       NAME               REASON
# ---------- ------------------ ----------------------------------------
# 2014/03/20 Kiyoshi Oishi      Initial Release
#=======================================================================
sub readResponseFile {
    my $file = $_[0];
    my $ref_res = $_[1];
    my $json = "";
    my $data = "";

    if (-e $file) {
        # ファイル読み込み
        open(FH, $file);
        $json = do {
            local $/;
            <FH>
        };
        close(FH);
        foreach my $key (keys(%$ref_res)) {
            $json =~ s/\$\{$key\}/$ref_res->{$key}/;
        }
    } else {
        # レスポンスを作成する。
        $json = createResponse($file, $ref_res);
    }

    # 読込んだファイルの内容をjson形式にデコード
    $data = decode_json($json);

    return $data;
}

#=======================================================================
# FuncName : createResponse
# Summary  : レスポンス作成
# Argument : (IN)  file        : 作成ファイル名(scalar)
#          : (IN)  ref_result  : 結果(ref_hash)
# Return   : 正常:0、異常:-1
# ----------------------------------------------------------------------
# DATE       NAME               REASON
# ---------- ------------------ ----------------------------------------
# 2014/03/27 Kiyoshi Oishi      Initial Release
#=======================================================================
sub createResponse {
    my $rc = ComConst::STS_OK;
    my $file       = $_[0];
    my $ref_result = $_[1];
    my $json       = "";

    # レスポンスを作成
    if ($rc == ComConst::STS_OK) {
        $json = decode_utf8(encode_json($ref_result));
    }

    return $json;
}

#=======================================================================
# FuncName : setResponsData
# Summary  : Webへのレスポンス
# Argument : (IN)  resJson  : 返却するデータ(ref_hash)
# Return   : 正常:0、異常:-1
# ----------------------------------------------------------------------
# DATE       NAME               REASON
# ---------- ------------------ ----------------------------------------
# 2014/03/20 Kiyoshi Oishi      Initial Release
#=======================================================================
sub setResponsData {
    my $resJson = $_[0];
    my $header  = $_[1];

    if ($header ne "") {
        print $header;
    } else {
        print "Content-type: application/json\n\n";
    }
    print encode_json($resJson);
}

#=======================================================================
# FuncName : setCtrlApiResponsData
# Summary  : Webへのレスポンス
# Argument : (IN)  resData  : 返却するデータ(ref_hash)
# Argument : (IN)  header   : レスポンスヘッダ(scalar)
# Return   : 正常:0、異常:-1
# ----------------------------------------------------------------------
# DATE       NAME               REASON
# ---------- ------------------ ----------------------------------------
# 2015/12/01 Kiyoshi Oishi      Initial Release
#=======================================================================
sub setCtrlApiResponsData {
    my $resData = $_[0];
    my $header  = $_[1];
    my $resStr  = "";

    if ($header ne "") {
        print $header;
    } else {
        print "Content-type: text/plain\n\n";
    }

    foreach my $key (keys(%$resData)) {
        if ($key ne "data") {
            if ($resStr eq "") {
                $resStr = "$key=$resData->{$key}";
            } else {
                $resStr .= "&$key=$resData->{$key}";
            }
        }
    }
    print $resStr;
}

#=======================================================================
# FuncName : getPostSendData
# Summary  : リクエストされたWebリモコンの値をシステムに送信する
# Argument : (IN) $data     : リクエストデータ(ref_hash)
# Return   : Nothing
# ----------------------------------------------------------------------
# DATE       NAME               REASON
# ---------- ------------------ ----------------------------------------
# 2015/11/27 Kiyoshi Oishi      Initial Release
#=======================================================================
sub getPostSendData {
    my $rc = ComConst::STS_OK;
    my $ref_postData = $_[0];
    my $postDataTemplate = $_[1];
    my $sendkey = $_[2];
    my $value = $_[3];
    my $auth = $_[4];
    my $targetView = $_[5];

    if ($auth ne "") {
        if ($targetView ne "") {
            $$ref_postData = sprintf($postDataTemplate, $sendkey, $targetView, $value, $auth);
        } else {
            $$ref_postData = sprintf($postDataTemplate, $sendkey, $value, $auth);
        }
    } else {
        $$ref_postData = sprintf($postDataTemplate, $sendkey, $value);
    }

    return $rc;
}

#=======================================================================
# FuncName : getPostSendDataViewLayout
# Summary  : リクエストされたWebリモコンの値をシステムに送信する
# Argument : (IN) $data     : リクエストデータ(ref_hash)
# Return   : Nothing
# ----------------------------------------------------------------------
# DATE       NAME               REASON
# ---------- ------------------ ----------------------------------------
# 2015/11/27 Kiyoshi Oishi      Initial Release
#=======================================================================
sub getPostSendDataViewLayout {
    my $rc = ComConst::STS_OK;
    my $ref_postData = $_[0];
    my $postDataTemplate = $_[1];
    my $sendkey = $_[2];
    my $value = $_[3];
    my $auth = $_[4];
    my $ref_option_params = $_[5];

    $$ref_postData = sprintf($postDataTemplate, $sendkey, $ref_option_params->{view}, $value, $auth);

    return $rc;
}

#=======================================================================
# FuncName : getDate
# Summary  : 日付取得(yyyymmddHHMMDD形式)
# Argument : Nothing
# Return   : yyyymmddHHMMDD形式の日付
# ----------------------------------------------------------------------
# DATE       NAME               REASON
# ---------- ------------------ ----------------------------------------
# 2014/04/16 Kiyoshi Oishi      Initial Release
#=======================================================================
sub getDate {
    my $format = ComConst::DEFAULT_DATE_FORMAT;
    my @dateTime = localtime(time);

    return sprintf($format, 1900+$dateTime[5], 1+$dateTime[4], $dateTime[3], $dateTime[2], $dateTime[1], $dateTime[0]);
}


#=======================================================================
# FuncName : trim
# Summary  : 最初および最後から空白文字を取り除く
# Argument : (IN)  str    : 処理文字列 (scalar)
# Return   : 取り除かれた文字列 (scalar)
# ----------------------------------------------------------------------
# DATE       NAME               REASON
# ---------- ------------------ ----------------------------------------
# 2014/03/07 Kiyoshi Oishi      Initial Release
#=======================================================================
sub trim {
    my $str = $_[0];

    return ltrim(rtrim($str));
}

#=======================================================================
# FuncName : ltrim
# Summary  : 先頭からの空白文字を取り除く
# Argument : (IN)  str    : 処理文字列 (scalar)
# Return   : 取り除かれた文字列 (scalar)
# ----------------------------------------------------------------------
# DATE       NAME               REASON
# ---------- ------------------ ----------------------------------------
# 2014/03/07 Kiyoshi Oishi      Initial Release
#=======================================================================
sub ltrim {
    my $str = $_[0];

    $str =~ s/^\s+(.*)/$1/;
    return $str
}

#=======================================================================
# FuncName : rtrim
# Summary  : 最後からの空白文字を取り除く
# Argument : (IN)  str    : 処理文字列 (scalar)
# Return   : 取り除かれた文字列 (scalar)
# ----------------------------------------------------------------------
# DATE       NAME               REASON
# ---------- ------------------ ----------------------------------------
# 2014/03/07 Kiyoshi Oishi      Initial Release
#=======================================================================
sub rtrim {
    my $str = $_[0];

    $str =~ s/\s+$//;
    return $str;
}

#=======================================================================
# FuncName : zlength
# Summary  : 全角半角文字数
# Argument : (IN)  str    : 処理文字列 (scalar)
# Return   : 文字数 (scalar)
# ----------------------------------------------------------------------
# DATE       NAME               REASON
# ---------- ------------------ ----------------------------------------
# 2022/10/11 Kiyoshi Oishi      Initial Release
#=======================================================================
sub zlength {
    my $decStr = decode('UTF-8', $_[0]);

    return length($decStr);
}

1;
