#!/usr/bin/perl
package ComLog;
#=======================================================================
# FileName : ComLog.pm
# Summary  : ログ定義
# ----------------------------------------------------------------------
# DATE       NAME               REASON
# ---------- ------------------ ----------------------------------------
# 2015/03/09 Kiyoshi Oishi      Initial Release
#=======================================================================
use strict;

use ComConst;

##### ログレベル #####
use constant NO_LOG      => 0;
use constant LOG_INFO    => 1;
use constant LOG_WARNING => 2;
use constant LOG_ERROR   => 4;
use constant LOG_FATAL   => 8;
use constant LOG_DEBUG   =>16;

our $_LOG_LEVEL = LOG_INFO | LOG_WARNING | LOG_ERROR;
our $_MODULE    = "";
our $_LOG_FILE  = "-";

#==============================================================================
# Function Name : log_init
# abstract      : ログ初期化
# function      : 1. 引数をそれぞれログ表示レベル、モジュール名、ログファイル名
#                    の変数に設定する。
#                 2. STS_OK を戻り値として終了する。
# call          : log_init($log_level, $module, $log_file)
# param         : $log_level  表示するログレベル
#                 $module     モジュール名
#                 $log_file   ログファイル名
# return        : 終了ステータス
#                   $STS_OK        正常終了
#==============================================================================
# Update History
#
# Date       Name            Reason for change
# ---------- --------------- --------------------------------------------------
# 2006/05/29 Hiroyuki Kouga  Initial Release
#==============================================================================
sub log_init {
    my ($log_level, $module, $log_file) = @_;

    if ($log_level ne "") {
        $_LOG_LEVEL = NO_LOG;
        my $tmpLevel;
        foreach my $level (split(/\|/, $log_level)) {
            if ($level eq "INFO") {
                $tmpLevel = LOG_INFO;
            }
            if ($level eq "WARNING") {
                $tmpLevel = LOG_WARNING;
            }
            if ($level eq "ERROR") {
                $tmpLevel = LOG_ERROR;
            }
            if ($level eq "FATAL") {
                $tmpLevel = LOG_FATAL;
            }
            if ($level eq "DEBUG") {
                $tmpLevel = LOG_DEBUG;
            }
            $_LOG_LEVEL = $_LOG_LEVEL | $tmpLevel;
        }
    }
    $_MODULE    = $module;
    $_LOG_FILE  = $log_file;

    return ComConst::STS_OK;
}

#==============================================================================
# Function Name : log_output
# abstract      : ログ出力関数
# function      : 1. ステータスを STS_OK に設定する。
#                 2. ログメッセージレベルがログ表示レベルより大きければ 7 へ。
#                 3. ログ文字列を以下のフォーマットで生成する。
#                      月 日 時:分:秒 モジュール名[PID]:[ログレベル] メッセージ
#                 4. ログファイルをオープンする。
#                    失敗したらステータスを STS_ERR_FILE とし、6 へ。
#                 5. 3 で生成したログ文字列をログファイルに追記する。
#                 6. 4 でログファイルがオープンされていたらクローズし、
#                    そのパーミッションを rw-rw-rw- に設定する。
#                 7. ステータスを戻り値として終了する。
# call          : log_output($log_level, $msg)
# param         : $log_level    -- ログを表示するレベル
#               : $msg          -- ログ文字列
# return        : $STS_OK       -- 正常終了
#               : $STS_ERR_FILE -- ファイルエラー
#==============================================================================
# Update History
#
# Date       Name            Reason for change
# ---------- --------------- --------------------------------------------------
# 2006/05/29 Hiroyuki Kouga  Initial Release
#==============================================================================
sub log_output {
    my $sts = ComConst::STS_OK;
    my $open;
    my $log_msg;
    my @tmp;
    my @month;
    my %log_level_str;
    my ($log_level, $msg) = @_;
    my $funcName = (caller 1)[3];
    $funcName =~ s/.*:://;

    #$open = false;
    $open = 0;

    if ($log_level & $_LOG_LEVEL) {       # ログ出力レベルと比較
        @month = ('Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun',
                            'Jul', 'Aug', 'Sep', 'Oct', 'Nov', 'Dec');
        %log_level_str = (0=>'', 1=>'Info', 2=>'Warning', 4=>'Error', 8=>'Fatal', 16=>'Debug');

        ## ログ文字列生成
        @tmp = localtime(time);

        $log_msg = sprintf("%s %02d %02d:%02d:%02d %s[%s]:[%s] %s:%s",
                            $month[$tmp[4]], $tmp[3], $tmp[2], $tmp[1], $tmp[0],
                            $_MODULE, $$, $log_level_str{$log_level}, $funcName, $msg);

        ## ログファイルオープン
        if (open(LOG, ">>$_LOG_FILE")) {
            $open = 1;
        } else {
            $sts = ComConst::STS_ERR_FILE;
        }

        ## ログ文字列出力
        if ($sts == ComConst::STS_OK) {
            printf(LOG "%s\n", $log_msg);
        }

        ## ログファイルクローズ
        if ($open) {
            close(LOG);
            chmod(0666, $_LOG_FILE);
        }
    }

    return $sts;
}

1;
###############################################################################
# End of File
###############################################################################
